(function(){
  const SB = window.SB_WEBINAR || {};

  const els = {
    shell: document.getElementById('sb-shell'),
    chat: document.getElementById('sb-chat'),
    toggleChat: document.getElementById('sb-toggle-chat'),
    collapseChat: document.getElementById('sb-collapse-chat'),
    openChatMobile: document.getElementById('sb-open-chat-mobile'),
    chatBody: document.getElementById('sb-chat-body'),
    chatInput: document.getElementById('sb-chat-input'),
    chatSend: document.getElementById('sb-chat-send'),
    statusPill: document.getElementById('sb-status-pill'),
    watchTime: document.getElementById('sb-watch-time'),
    video: document.getElementById('sb-video'),
    videoOverlay: document.getElementById('sb-video-overlay'),
    identMini: document.getElementById('sb-ident-mini'),
  };

  function safeText(s){
    return (s ?? '').toString().replace(/[\u0000-\u001F\u007F]/g,'').trim();
  }

  function fmtTime(seconds){
    seconds = Math.max(0, Math.floor(seconds || 0));
    const m = Math.floor(seconds / 60);
    const s = seconds % 60;
    return `${m}:${String(s).padStart(2,'0')}`;
  }

  function setStatus(text, ok){
    if (!els.statusPill) return;
    els.statusPill.textContent = text;
    if (ok){
      els.statusPill.style.background = 'rgba(166,255,0,.14)';
      els.statusPill.style.borderColor = 'rgba(166,255,0,.26)';
      els.statusPill.style.color = 'var(--sb-green)';
    } else {
      els.statusPill.style.background = 'rgba(255,255,255,.06)';
      els.statusPill.style.borderColor = 'rgba(255,255,255,.14)';
      els.statusPill.style.color = 'var(--sb-silver)';
    }
  }

  function nowIso(){
    return new Date().toISOString();
  }

  function getIdent(){
    return {
      email: SB.ident?.email || '',
      phone: SB.ident?.phone || '',
      ip: SB.ident?.ip || '',
      ip_observed: SB.ident?.ip_observed || ''
    };
  }

  function renderIdentMini(){
    const i = getIdent();
    const bits = [];
    if (i.email) bits.push(`email: ${i.email}`);
    if (i.phone) bits.push(`phone: ${i.phone}`);
    if (i.ip) bits.push(`ip: ${i.ip}`);
    els.identMini.textContent = bits.length ? `Attached: ${bits.join(' • ')}` : 'Tip: pass ?email= and/or ?phone= in the URL for best matching.';
  }

  function addMsg(who, text){
    text = safeText(text);
    if (!text) return;

    const wrap = document.createElement('div');
    wrap.className = `sb-msg ${who === 'me' ? 'sb-me' : 'sb-alex'}`;

    const meta = document.createElement('div');
    meta.className = 'sb-msg-meta';
    const left = document.createElement('div');
    const right = document.createElement('div');
    left.innerHTML = `<b>${who === 'me' ? 'You' : 'Alex'}</b>`;
    right.textContent = new Date().toLocaleTimeString([], {hour:'2-digit', minute:'2-digit'});
    meta.appendChild(left);
    meta.appendChild(right);

    const body = document.createElement('div');
    body.className = 'sb-msg-text';
    body.textContent = text;

    wrap.appendChild(meta);
    wrap.appendChild(body);

    els.chatBody.appendChild(wrap);
    els.chatBody.scrollTop = els.chatBody.scrollHeight;
  }

  function toggleDesktopCollapse(){
    if (!els.shell) return;
    els.shell.classList.toggle('sb-chat-collapsed');
  }

  function openMobileChat(){
    if (!els.chat) return;
    els.chat.classList.add('sb-chat-open');
    setTimeout(()=>els.chatInput?.focus(), 50);
  }

  function closeMobileChat(){
    if (!els.chat) return;
    els.chat.classList.remove('sb-chat-open');
  }

  function isMobile(){
    return window.matchMedia('(max-width: 980px)').matches;
  }

  function bindUI(){
    els.toggleChat?.addEventListener('click', ()=>{
      if (isMobile()) return openMobileChat();
      toggleDesktopCollapse();
    });

    els.collapseChat?.addEventListener('click', ()=>{
      if (isMobile()) return closeMobileChat();
      toggleDesktopCollapse();
    });

    els.openChatMobile?.addEventListener('click', openMobileChat);

    els.chatSend?.addEventListener('click', ()=>{
      const text = safeText(els.chatInput?.value);
      if (!text) return;
      els.chatInput.value = '';
      addMsg('me', text);
      sendChat(text).catch(()=>{
        addMsg('alex', 'Hmm — I had trouble sending that. Try again in a second.');
      });
    });

    els.chatInput?.addEventListener('keydown', (e)=>{
      if (e.key === 'Enter'){
        e.preventDefault();
        els.chatSend?.click();
      }
    });

    // Tap outside to close on mobile
    document.addEventListener('click', (e)=>{
      if (!isMobile()) return;
      if (!els.chat.classList.contains('sb-chat-open')) return;
      const target = e.target;
      if (!target) return;
      const insideChat = els.chat.contains(target);
      const onButton = els.openChatMobile?.contains(target);
      if (!insideChat && !onButton) closeMobileChat();
    });
  }

  async function postJson(path, body){
    const url = `${(SB.apiBase || '').replace(/\/$/,'')}${path}`;

    const res = await fetch(url, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-SB-Webhook-Token': SB.webhookToken || ''
      },
      body: JSON.stringify(body)
    });

    // Don't assume CORS is set; if blocked, browser will throw.
    if (!res.ok){
      const t = await res.text().catch(()=> '');
      throw new Error(`HTTP ${res.status}: ${t}`);
    }
    return await res.json().catch(()=> ({}));
  }

  async function sendChat(message){
    const payload = {
      webinar_id: SB.webinar_id,
      message: message,
      ident: getIdent(),
      page_url: SB.pageUrl,
      client_ts: nowIso()
    };

    setStatus('Sending…', true);
    const out = await postJson('/webhook/webinar/chat-inbound', payload);
    setStatus('Connected', true);

    // Optional: if backend returns an immediate assistant message
    if (out && typeof out.reply === 'string' && out.reply.trim()){
      addMsg('alex', out.reply.trim());
    } else {
      // Friendly acknowledgement (don’t spam)
      // Backend will typically respond later via web chat polling; for now we stay quiet.
    }

    return out;
  }

  let lastProgressSentAt = 0;
  let lastSentWatch = -1;

  async function sendProgress(watchSeconds){
    const now = Date.now();
    const intervalMs = Math.max(3, (SB.progressIntervalSeconds || 10)) * 1000;
    if ((now - lastProgressSentAt) < intervalMs) return;

    const ws = Math.max(0, Math.floor(watchSeconds || 0));
    if (ws === lastSentWatch) return;

    lastProgressSentAt = now;
    lastSentWatch = ws;

    const payload = {
      webinar_id: SB.webinar_id,
      watch_seconds: ws,
      ident: getIdent(),
      page_url: SB.pageUrl,
      client_ts: nowIso()
    };

    try {
      await postJson('/webhook/webinar/progress', payload);
      setStatus('Connected', true);
    } catch (e){
      setStatus('Offline', false);
    }
  }

  function bindVideo(){
    if (!els.video) return;

    const hasSrc = els.video.querySelector('source')?.getAttribute('src');
    if (!hasSrc){
      els.videoOverlay.style.display = 'flex';
    } else {
      els.videoOverlay.style.display = 'none';
    }

    els.video.addEventListener('timeupdate', ()=>{
      const t = els.video.currentTime || 0;
      if (els.watchTime) els.watchTime.textContent = fmtTime(t);
      sendProgress(t);
    });

    els.video.addEventListener('play', ()=>{
      // Send an immediate progress tick on play
      sendProgress(els.video.currentTime || 0);
    });

    els.video.addEventListener('ended', ()=>{
      // Final progress tick
      sendProgress(els.video.duration || els.video.currentTime || 0);
    });

    // Send periodic progress even if timeupdate throttles on some browsers
    setInterval(()=>{
      if (document.hidden) return;
      if (els.video.paused) return;
      sendProgress(els.video.currentTime || 0);
    }, 5000);
  }

  function boot(){
    bindUI();
    bindVideo();
    renderIdentMini();

    setStatus('Connected', true);
    addMsg('alex', "Hey — I’m Alex with SalesBeast. If you’ve got questions while you watch, ask here anytime.");
  }

  boot();
})();
